/*
 * MPXAVR.c, Version 2.2
 *
 * Copyright 2008 by Christian Persson. 
 *
 * ATTiny 25 as monoflop connecting the FASST T6EXP TX modul to the MPX Cockpit SX.
 *
 * As usual, Cockpit SX channel 4 is used for the motor throttle.
 * The adapter maps channel 4 to TX channel 3 and channel 3 to TX channel 4.
 * It generates a -100% signal on channel 8, thus providing the channel 3 failsafe setting.
 *
 * PB0 (pin5) is output, PB1 (pin6) is power down button input, PB2 (pin7) is PPM signal input.
 *
 * Timer 1 samples PPM input signal, timer 0 controls output signal timing.
 *
 * Version 1.11: Sync pulse detect time reduced to 3.2ms, motor off reduced to 0.95ms
 * Version 2.0: Timer resolution reduced to 1s to avoid IRQ overload causing jitter
 * Version 2.1: PB5 pullup resistor, Confirm PB0 output in main loop
 * Version 2.2: Output delay reduced to 2ms; 8th input channel enabled
 */


#include <stdint.h>
#include <inttypes.h>
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/wdt.h> 

// F_CPU = 8000000 (8Mhz) is defined in the makefile
// timer prescaler /8, all durations in s
#define  Pulse420 420 
#define  Pulse440 440
#define  Pulse400 400	
#define  MotorOff 960	
#define  Neutral 1500	
#define	 Latency 2

volatile uint16_t Channel[9];							// channel pulse durations
volatile uint16_t LastChannel;	

register uint8_t T0lo asm ("r2");						// current timer0 value low byte
register uint8_t T0hi asm ("r3");						// current timer0 value high byte
register uint8_t T1lo asm ("r4");						// current timer1 value low byte
register uint8_t T1hi asm ("r5");						// current timer1 value high byte
register uint8_t Action asm ("r6");						// prepared TCCR0A value
register uint8_t InSync asm ("r7");						// input sequence sync flag
register uint8_t OutSync asm ("r8");					// output sequence sync flag
register uint8_t T0ovf asm ("r9");						// timer0 overflow counter
register uint8_t T1ovf asm ("r10");						// timer1 overflow counter
register uint8_t INCH asm ("r11");						// input channel counter
register uint8_t OUTCH asm ("r12");						// output channel counter



// INPUT: overflow interrupt is used to extend timer1 to 16 bits
ISR(TIM1_OVF_vect)						
{
	if (++T1ovf > 11) InSync = 1;						// sync pulse detection
}

// INPUT: falling edge on Pin 7 triggers INT0 to measure input pulse duration
ISR(INT0_vect)
{
	TCCR1 = 0;											// stop timer
	if (TCNT1 < 2 || (TIFR & 0x04) != 0) {				// missed overflow interrupt?
		T1ovf++;
		TIFR |= 0x04;									// clear IRQ flag
	}
	T1lo = TCNT1;										// register duration
	T1hi = T1ovf;										// 16 bit
	T1ovf = 0;
	TCNT1 = 0;											// restart timer
	TCCR1 = 0x04;
	InSync++;											// flag is 2 following sync pulse, otherwise 4 
}

// OUTPUT: Timer0 compare match interrupt
ISR(TIM0_COMPA_vect)
{	
	if (T0ovf > 0) {
		T0ovf--;
		if (T0ovf == 0) TCCR0A = Action; 				// action on next compare match 
	} else {											// time has expired
		TCNT0 = 0;										// reset timer
		OCR0A = T0lo;									// load prepared compare setting
		T0ovf = T0hi;									// 16 bit
		OutSync++;										// flag
	}
}


int main (void)
{

	CLKPR = 0x80;										// enable set clock prescaler
	CLKPR = 0x00;										// set cpu clock prescaler = 1
			
	WDTCR = 0x09;										// watchdog enable 32ms

	DDRB = 0x01;										// PB0 output
	PORTB = 0x36;										// PB0 low, activate PB1, PB2, PB4, PB5 pullup resistors

														// default timing
	// /* Cockpit SX													
	Channel[0] = Neutral;
	Channel[1] = Neutral;
	Channel[2] = Neutral;
	Channel[3] = MotorOff;
	Channel[4] = Neutral;
	Channel[5] = Neutral;
	Channel[6] = Neutral;
	Channel[7] = MotorOff;
	Channel[8] = Neutral;
	//*/ 
	/* Graupner
	Channel[0] = MotorOff;
	Channel[1] = Neutral;
	Channel[2] = Neutral;
	Channel[3] = Neutral;
	Channel[4] = Neutral;
	Channel[5] = Neutral;
	Channel[6] = Neutral;
	Channel[7] = MotorOff;
	Channel[8] = Neutral;
	*/
	MCUCR = 0x02;										// INT0 on falling edge on PB2
	GIMSK = 0x40;										// INT0 enable

	TIMSK = 0x14;										// OCIE0A, TOIE1 enable
	TCCR1 = 0x04;										// start timer1 (input)

	sei();
	
	// infinite main loop
	while(1)	{
		
		DDRB = 0x01;									// PB0 output
		if (InSync == 2) {								// sync pulse detected
			T0ovf = 8;									// output sequence delay approx. 2ms 
			TCNT0 = 0;									// start timer0 (output)
			TCCR0B = 0x02;
			INCH = 0;
			OUTCH = 0;
			OutSync = 1;
			InSync = 3;
		} else if (InSync > 3) {						// record pulse duration
			LastChannel = T1hi;							// max 8 channels 
			if (INCH < 8) Channel[INCH] = (LastChannel << 8) + T1lo - 3;
			INCH++;
			InSync = 3;
		}

		if (OutSync == 1) {								// prepare for high pulse
			if (OUTCH == 0) {
				if ((PINB & 0x02) != 0) {				// regular mode
					T0lo = (Pulse420 - Latency) & 0x00FF;
					T0hi = (Pulse420 - Latency) >> 8;
				} else {								// power down mode
					T0lo = (Pulse440 - Latency) & 0x00FF;
					T0hi = (Pulse440 - Latency) >> 8;
				}
			} else if (OUTCH == 1) {
				T0lo = (Pulse420 - Latency) & 0x00FF;
				T0hi = (Pulse420 - Latency) >> 8;			
			} else {
				T0lo = (Pulse400 - Latency) & 0x00FF;
				T0hi = (Pulse400 - Latency) >> 8;			
			}
			Action = 0xC0;								// action: Set OC0A on Compare Match
			OutSync = 2;
	
		} else if (OutSync == 3) {						// prepare for low pulse
			if (OUTCH < 9) {							// sync phase after 9th high pulse
				switch (OUTCH) {
					// /* Multiplex
					case 0:
						if ((PINB & 0x02) != 0) {		// power up mode
							T0lo = (Channel[0] - Pulse420 - Latency) & 0x00FF;
							T0hi = (Channel[0] - Pulse420 - Latency) >> 8;
						} else {						// power down mode
							T0lo = (Channel[0] - Pulse440 - Latency) & 0x00FF;
							T0hi = (Channel[0] - Pulse440 - Latency) >> 8;
						}
						break;
					case 1:
						T0lo = (Channel[1] - Pulse420 - Latency) & 0x00FF;
						T0hi = (Channel[1] - Pulse420 - Latency) >> 8;
						break;
					
					// swap channels 3, 4 (comment out if no swapping needed)
					case 2:								
						T0lo = (Channel[3] - Pulse400 - Latency) & 0x00FF;
						T0hi = (Channel[3] - Pulse400 - Latency) >> 8;
						break;
					case 3:								
						T0lo = (Channel[2] - Pulse400 - Latency) & 0x00FF;
						T0hi = (Channel[2] - Pulse400 - Latency) >> 8;
						break;
					// end swap channels 3, 4 (comment out if no swapping needed) 	
					
					default:
						T0lo = (Channel[OUTCH] - Pulse400 - Latency) & 0x00FF;
						T0hi = (Channel[OUTCH] - Pulse400 - Latency) >> 8;			
						break;
					// end Multiplex */
					
					/* Graupner	(swap channels 1, 3)
					case 0:
						if ((PINB & 0x02) != 0) {		// power up mode
							T0lo = (Channel[2] - Pulse420 - Latency) & 0x00FF;
							T0hi = (Channel[2] - Pulse420 - Latency) >> 8;
						} else {						// power down mode
							T0lo = (Channel[2] - Pulse440 - Latency) & 0x00FF;
							T0hi = (Channel[2] - Pulse440 - Latency) >> 8;
						}
						break;
					case 1:
						T0lo = (Channel[1] - Pulse420 - Latency) & 0x00FF;
						T0hi = (Channel[1] - Pulse420 - Latency) >> 8;
						break;
					case 2:								
						T0lo = (Channel[0] - Pulse400 - Latency) & 0x00FF;
						T0hi = (Channel[0] - Pulse400 - Latency) >> 8;
						break;
					
					default:
						T0lo = (Channel[OUTCH] - Pulse400 - Latency) & 0x00FF;
						T0hi = (Channel[OUTCH] - Pulse400 - Latency) >> 8;			
						break;
					end Graupner */
				}
				Action = 0x80;							// action: Clear OC0A on Compare Match
				OUTCH++;
			}
			OutSync = 4;

		} else if (OutSync > 4) {
			if (OUTCH < 9) {
				OutSync = 1;							// flag for another high pulse
			} else {
				TCCR0B = 0x00;							// stop timer0 during sync phase
				Channel[7] = MotorOff;					// failsafe preset
			}
		}
		if (T0lo == 0) T0lo = 1;						// avoid compare match blocking 

		wdt_reset();									// reset watchdog
	}

	return(0);
}
